# cdd.py - Data handling routines developed by Optibrium for integration  of CDD Vault with StarDrop software
# Use of this code is subject to the terms of the StarDrop License Agreement and use, copying or redistribution of
# this file for use by other persons or organisations is prohibited
# Copyright(C) Optibrium Ltd 2022
# stardrop-support@optibrium.com
#

"""Module to control access to the CDD Vault from StarDrop"""

import requests
from requests.exceptions import HTTPError
import json
import time
import sys
import app
import PythonWidgets
import cddconfig
import CSVHelper
from SDFileHelpers import createDatasetFromSDFile
import shutil
import tempfile
import struct
import threading
import logging

SCRIPT_VERSION = "CDD Vault Version 5.0"

TOKEN = cddconfig.get_token()
USER_CANCELLED = "Operation cancelled by user"

logger = None

try:
    import refreshscript

    stardrop = sys.modules["__main__"].appui

    def refresh_CDD_search(method, params):
        """Run a search and supply the information to StarDrop to reform a refresh operation.
        @param: method: Should be 'CDD'.
        @param: params: The parameters to use when running the search.
        """
        if method != "CDD":
            raise Exception("method is not CDD")
        else:
            try:
                logger.info('refreshing CDD')

                vault_id = int(params.get("vault", ""))
                search_id = int(params.get("search", ""))
                projects = params.get("projects", "").split(',')
                action = params.get("action", "")
                protocol_ids = params.get("protocol_ids", "").split(',')
                size = params.get("size", "")

                params = (vault_id, search_id, projects, protocol_ids, size)

                dataset, _ = _determine_function(stardrop, action, params)

                return dataset.pack()
            except Exception as e:
                print(str(e))
                raise

    print('adding CDD refresh')
    refreshscript.setRefreshFunction(refresh_CDD_search)
except Exception as e:
    print(str(e))


def setup_cdd_logger():
    global logger
    if not logger:
        log_console_format = "[%(asctime)s] - [%(levelname)s]: %(message)s"
        logger = logging.getLogger(__name__)
        logger.setLevel(cddconfig.LOGGING_LEVEL)
        if not logger.handlers:
            console_handler = logging.FileHandler(cddconfig.LOGGER_FILE)
            console_handler.setLevel(cddconfig.LOGGING_LEVEL)
            console_handler.setFormatter(logging.Formatter(fmt=log_console_format,
                                                           datefmt='%d-%b-%y %H:%M:%S'))

            logger.addHandler(console_handler)
    logger.info(SCRIPT_VERSION)
    logger.info(f"Server: {cddconfig.url}")


setup_cdd_logger()


LOCK = threading.Lock()


class ImageCollector(threading.Thread):
    def __init__(self, vault_id, molecule, batch_id, protocol_id, url, results, index, size):
        threading.Thread.__init__(self)
        self._vault_id = vault_id
        self._molecule = molecule
        self._batch_id = batch_id
        self._protocol_id = protocol_id
        self._results = results
        self._index = index
        self._url = url
        self.size = size

    def run(self):
        params = {'size': self.size}
        # Grab the plot data from CDD Vault
        results = perform_plot_call(self._vault_id,
                                    self._batch_id,
                                    self._protocol_id,
                                    params)
        if results:
            # Convert the plot data into an image
            entries = create_image_data_entry(results, self._url)
            if entries:
                with LOCK:
                    if self._molecule not in self._results:
                        self._results[self._molecule] = {}
                        self._results[self._molecule][self._protocol_id] = (entries[0],
                                                                            entries[1],
                                                                            self._index)
            else:
                logger.debug("No image data entries created")
        else:
            logger.debug("Failed to retrieve image for: " +
                         f"vault: {self._vault_id} - batch: {self._batch_id} - protocol: {self._protocol_id}")


def change_token(appui):
    """Respond to a user request to change the API token being used.
    @params appui: The StarDrop application interface.
    """
    global TOKEN
    new_token = appui.query("CDD Token", "Please enter your CDD token:", "")
    if new_token == '':
        logger.debug("Token has not been set")
        return
    TOKEN = new_token
    logger.debug(f"Token now set to {TOKEN}")
    cddconfig.set_token(TOKEN)


def _get_headers(token):
    """Construct a set of headers to pass to each request to the server,
    @param: token: The API token to be used to validate the call.
    @return: A dictionary of headers.
    """
    headers = {"Content-Type": "application/json"}
    headers["User-Agent"] = f"Optibrium StarDrop {SCRIPT_VERSION}"
    headers["X-CDD-Token"] = token
    return headers


def run_search(appui):
    """Run a saved search and retrieve the data from CDD Vault
    @params appui: The StarDrop application interface.
    """
    # Have the user select the function they want to use
    option_dialog = PythonWidgets.OptionSelector(True)
    option_dialog.setWindowTitle("Choose Search Function")
    option_dialog.setLabel("Select one option:")
    option_dialog.setAvailableItems(["Download data only",
                                     "Download with response curve urls",
                                     "Download with response curve plots"])
    option_dialog.setSelectedItems(["Download data only"])
    dataset, dataset_id = None, None
    if cddconfig.ACCESS_RESPONSE_CURVES:
        if option_dialog.run():
            action = option_dialog.getSelectedItem()
            logger.debug(f"Search Option Selected: {action}")
            dataset, dataset_id = _determine_function(appui, action)
    else:
        logger.debug("Response curves not required. Executing data only.")
        action = "Download data only"
        dataset, dataset_id = _determine_function(appui, action)

    if not dataset and not dataset_id:
        return
    try:
        if dataset:
            appui.exportDS(dataset_id, dataset.pack())
            logger.debug(f"Dataset {dataset_id} Exported")
            return
        else:
            logger.debug("Dataset is empty after full execution")
            appui.showError("Unable to run the search against the current project(s)")
            return
    except Exception as e:
        logger.exception(f"Dataset export was unsuccessful: {str(e)}")
        logger.debug("Dataset is created, but incompatible with export.")
        appui.showError("Unable to execute chosen saved search.")


def _determine_function(appui, action, refresh_params=None):
    """Perform the appropriate functionality for the chosen action.
    @params appui: The StarDrop application interface.
    @params action: The action for the script to perform.
    @params refresh_params: The parameters provided by the refresh script.
    @return: The constructed dataset and the id of the dataset.
    """
    if action == "Download data only":
        logger.debug(f"{action} feature selected")
        dataset, dataset_id, refresh_values = _retrieve_search_data(appui,
                                                                    "Download Data Only",
                                                                    refresh_params=refresh_params)
        refresh_values['action'] = action
        if (dataset):
            dataset.addRefreshKey(refresh_values)
        return dataset, dataset_id
    elif action == "Download with response curve urls":
        logger.debug(f"{action} feature selected")
        search_data, refresh_values = _retrieve_search_data(appui,
                                                            "Download with Response Curve URLs",
                                                            False,
                                                            refresh_params)
        refresh_values['action'] = action
        if search_data[0]:
            dataset, dataset_id, extra_refresh_values = _download_curve_data(appui,
                                                                             *search_data,
                                                                             get_plots=False,
                                                                             refresh_params=refresh_params)
            if extra_refresh_values:
                refresh_values.update(extra_refresh_values)
            if (dataset):
                dataset.addRefreshKey(refresh_values)
            return dataset, dataset_id
        else:
            logger.debug("Retrieving search data unsuccessful.")
            return None, None
    elif action == "Download with response curve plots":
        logger.debug(f"{action} feature selected")
        search_data, refresh_values = _retrieve_search_data(appui,
                                                            "Download with Response Curve Plots",
                                                            False,
                                                            refresh_params)
        refresh_values['action'] = action
        if search_data[0]:
            dataset, dataset_id, extra_refresh_values = _download_curve_data(appui,
                                                                             *search_data,
                                                                             get_plots=True,
                                                                             refresh_params=refresh_params)
            if extra_refresh_values:
                refresh_values.update(extra_refresh_values)
            if (dataset):
                dataset.addRefreshKey(refresh_values)
            return dataset, dataset_id
        else:
            logger.debug("Retrieving search data unsuccessful.")
            return None, None
    else:
        logger.debug(f"An invalid action was selected: {action}")
        return None, None


def _retrieve_search_data(appui, action, data_only=True, refresh_params=None):
    """Retrieve and download the data from a saved search on CDD Vault.
    @params appui: The StarDrop application interface.
    @params action: The action for the script to perform.
    @params data_only: If true export the dataset, if false return the data required for a plot call.
    @params refresh_params: The parameters provided by the referesh script.
    @returns: If the dataset is exported None. If it isn't the data required to make a plot call.
    """
    if not _connect_to_vault(appui):
        return None

    if appui:
        progress = PythonWidgets.SimpleProgress()
        progress.setWindowTitle("Running search")
        progress.show()

    # If using the refresh script assign the variables.
    if refresh_params:
        logger.debug("Assigning values via refresh parameters")
        vault_id_code = refresh_params[0]
        search_id_code = refresh_params[1]
        project_ids = refresh_params[2]
        dataset_title = ""
    # If not retrieve the variables from CDD
    else:
        logger.debug("Assigning values via user selection")
        # Get Vault ID
        vault_id = _get_vault_ID(progress, 10)
        if not vault_id:
            appui.showError("No vaults found")
            logger.debug("Vault ID is invalid.")
            return None, None
        elif vault_id == USER_CANCELLED:
            logger.debug("User cancelled vault selection.")
            return None, None
        vault_id_code = vault_id[0]
        logger.debug(f"Vault ID: {vault_id}")

        # Get Search ID
        search_id = _get_search_ID(vault_id, progress, 20)
        if not search_id:
            appui.showError("No searches found")
            logger.debug("Search ID is invalid.")
            return None, None
        elif search_id == USER_CANCELLED:
            logger.debug("User cancelled search selection.")
            return None, None
        logger.debug(f"Search ID: {search_id}")
        search_id_code = search_id[0]
        dataset_title = search_id[1]

        # Get Project IDs
        project_ids = _get_project_IDs(vault_id, progress, 30)
        if not project_ids:
            appui.showError("No projects available to search")
            logger.debug("Project ID is invalid.")
            return None, None
        elif project_ids == USER_CANCELLED:
            logger.debug("User cancelled project selection.")
            return None, None

    if appui:
        progress.setWindowTitle("Execute search")
        progress.setProgress(40)

    # Execute Search
    result = _execute_search(project_ids, vault_id_code, search_id_code)

    if not result:
        appui.showError("Error conducting search")
        logger.debug("Invalid results from search call.")
        return None, None
    # Wait for Data to process
    ready, export_id = _check_search_status(appui, result, vault_id_code)

    if appui:
        progress.setWindowTitle("Constructing data set")
        progress.setProgress(50)

    # Retrieve Data and Create Dataset
    if ready:
        logger.debug("Data from API is ready to process.")
        dataset = _construct_dataset_from_search(appui,
                                                 vault_id_code,
                                                 export_id)
    else:
        logger.warning("Data from API is not ready to process. Returning.")
        return None, None

    refresh_values = {}
    refresh_values['function'] = 'CDD'
    refresh_values['vault'] = repr(vault_id_code)
    refresh_values['search'] = repr(search_id_code)
    refresh_values['projects'] = ','.join(map(str, project_ids))
    logger.debug(f"Refresh values set: {refresh_values}")

    progress.hide()
    if data_only:
        logger.debug("Returning data only dataset")
        return dataset, dataset_title, refresh_values
    else:
        logger.debug("Returning data dataset")
        return (dataset,
                project_ids,
                vault_id_code,
                progress,
                dataset_title), refresh_values


def _download_curve_data(appui, dataset, project_ids, vault_id,
                         progress, dataset_title,
                         get_plots=False, refresh_params=None):
    """Download the response curve data from CDD Vault.
    @params appui: The StarDrop application interface.
    @params dataset: A constructed StarDrop dataset.
    @params project_ids: A list of project ids to filter the molecule search.
    @params vault_id: The id of the selected vault.
    @params progress: A progress dialog.
    @params dataset_title: The title of the dataset.
    @params get_plots: True if we want to download the plot images, False if not.
    @params: refresh_params: Parameters set during initial download and used to refresh the data.
    @returns: The newly constructed dataset and the title of the dataset.
    """
    progress.show()

    # Get Molecule Data
    column_index = dataset.columnIndex(cddconfig.CURVE_KEY_COLUMN_NAME)
    if column_index < 0:
        appui.showError("Unable to locate Molecule ID." +
                        f" Ensure that the saved search includes the {cddconfig.CURVE_KEY_COLUMN_NAME} field.\n" +
                        "StarDrop will load the data set without the curves")
        logger.error(f"No column in the dataset with the title '{cddconfig.CURVE_KEY_COLUMN_NAME}'." +
                     "This is required to retrieve curve data.")
        progress.hide()
        return dataset, dataset_title, refresh_params

    molecule_names = []
    num_rows = dataset.rowSz()

    molecule_dict = {}
    for i in range(num_rows):
        molecule_name = dataset.stringval(column_index, i)
        molecule_names.append(molecule_name)
        molecule_dict[molecule_name] = i

    project_ids_str = [str(id) for id in project_ids]

    # Filter the Molecule API call with the following parameters
    params = {}
    unique_molecule_names = set([m if m.count('-') == 1 else m[:m.rfind('-')] for m in molecule_names])
    unique_molecule_names = list(unique_molecule_names)
    params['projects'] = ",".join(project_ids_str)
    params["async"] = True

    # Get Molecule Info
    molecule_info = _get_molecules(vault_id, params)
    if not molecule_info:
        logger.debug("Molecule info from API is invalid.")
        progress.hide()
        appui.showError("Problem accessing curve data\nStarDrop will load the data set without the curves")
        return dataset, dataset_title, {}

    # Get molecule ids
    molecule_ids = []
    for m in molecule_info["objects"]:
        molecule_ids.append(m["id"])

    logger.debug(f"Molecule IDs: {molecule_ids}")

    # Get Protocols
    if refresh_params:
        logger.debug("Refresh parameters available, using them to retrieve protocol data.")
        logger.debug(f"Refresh parameters for protocols: {refresh_params[3]}")
        protocol_ids, protocol_info = _list_protocols(progress,
                                                      70,
                                                      vault_id,
                                                      molecule_ids,
                                                      [int(pid) for pid in refresh_params[3]])
    else:
        logger.debug("Refresh parameters are not available. User must select desired protocols.")
        protocol_ids, protocol_info = _list_protocols(progress,
                                                      70,
                                                      vault_id,
                                                      molecule_ids)

    if not protocol_ids:
        appui.showError("No protocols found.\nStarDrop will load the data set without the curves")
        logger.debug("Protocol IDs are invalid.")
        progress.hide()
        return dataset, dataset_title, {}
    elif protocol_ids == USER_CANCELLED:
        logger.debug("User cancelled protocol selection.")
        progress.hide()
        return None, None, None

    entry_data = []
    all_batch_ids = {}
    # Plot Calls
    for molecule in molecule_info["objects"]:
        # Get the batch id from the molecule data
        batch_ids = _get_batch_IDs_for_molecule(molecule)
        if batch_ids:
            all_batch_ids[molecule["name"]] = batch_ids
    logger.debug(f"All Batch IDs: {all_batch_ids}")

    plot_number = num_rows * len(protocol_ids)
    logger.debug(f"{plot_number} plot images required, across {len(protocol_ids)} protocols.")
    if get_plots and plot_number > cddconfig.HIGH_PLOT_RETRIEVAL_VALUE:
        logger.debug("Large amounts of molecules detected, checking if the user wants to get plots for them all.")
        w = PythonWidgets.Warning()
        w.setWindowTitle("Long Process Time")
        w.setMessage(f"You have chosen to download up to {plot_number} images.\n" +
                     "This may take several minutes, are you sure you want to proceed?")
        w.setOKText("Yes")
        w.setCancelText("No")

        if not w.run():
            logger.debug("User has decided that they no longer want to retrieve plot data.")
            get_plots = False

    image_size = 'small'
    if refresh_params and get_plots:
        image_size = refresh_params[4]
        logger.debug(f"Refresh parameters available, using them to set the image size to {image_size}")
    else:
        if get_plots:
            size_selector = PythonWidgets.OptionSelector(True)
            size_selector.setWindowTitle("Select Plot Image Size")
            size_selector.setLabel("Select one option:")
            size_selector.setAvailableItems(["Small",
                                             "Medium",
                                             "Large"])
            size_selector.setSelectedItems(["Small"])
            if size_selector.run():
                image_size = size_selector.getSelectedItem()
                logger.debug("Refresh parameters not available, the user is selecting the image size.")

    # Create parameters to retrieve image data
    image_parameters = []
    logger.debug("Image parameters collection has started.")
    logger.debug(f"Batch IDS: {all_batch_ids}")
    logger.debug(f"Protocol_IDs: {protocol_ids}")
    logger.debug(f"Protocol_Info: {protocol_info}")
    logger.debug(f"Molecule Dict: {molecule_dict}")
    for molecule in all_batch_ids:
        batches = all_batch_ids[molecule]
        for protocol in protocol_ids:
            protocol_name = "CDD"
            for p in protocol_info:
                if protocol_info[p] == protocol:
                    protocol_name = p
                    break

            for batch in batches:
                if batch[1] in molecule_dict:  # check molecule-batch name exists in data
                    url = f'<a href=\"https://app.collaborativedrug.com/vaults/{vault_id}/'
                    url += f'specified_batches/{batch[0]}/protocols/{protocol}/dose_response_plot\">'
                    url += f'DRC: {protocol_name}</a>'
                    image_parameters.append((vault_id,
                                             batch[0],
                                             protocol,
                                             protocol_name,
                                             batch[1],
                                             molecule_dict[batch[1]],
                                             url,
                                             image_size))
    logger.debug("Image parameters collection has ended.")

    if appui:
        progress.setWindowTitle("Exporting dataset")
        progress.setProgress(95)

    # Get the plot images and url
    if get_plots:
        logger.debug("Generating plot images and url link data entries.")
        entry_data = _generate_images(progress, image_parameters)
        logger.debug("Plot images and url link data entries are generated.")
    # Just get the url
    else:
        logger.debug("Generating plot url link data entries")
        data_list = []

        for _, _, protocol_, _, _, index_, url, _ in image_parameters:
            entry = app.Dataset.string(url)
            if entry:
                data_list.append((protocol_,
                                  index_,
                                  entry))
            else:
                data_list.append((protocol_,
                                  index_,
                                  app.Dataset.createInvalidEntry(app.string)))

        entry_data = data_list
        logger.debug("Plot url link data entries are generated.")

    refresh_values = {}
    refresh_values['protocol_ids'] = ','.join(map(str, protocol_ids))
    refresh_values['size'] = image_size
    logger.debug(f"Refresh parameters assigned: {refresh_values}")

    # Add new columns to the dataset
    if entry_data:
        logger.debug("Creating columns for the new data entries.")
        selected_protocols = {}
        logger.debug("Determining selected protocols.")
        for key in protocol_info:
            if protocol_info[key] in protocol_ids:
                selected_protocols[key] = protocol_info[key]
        logger.debug(f"Determined selected protocols: {selected_protocols}")

        new_dataset = _create_columns(dataset,
                                      selected_protocols,
                                      entry_data,
                                      get_plots)

        logger.info("Downloading of the curve data successful. Returning updated dataset.")
        return new_dataset, dataset_title, refresh_values
    else:
        logger.info("Downloading of curve data failed. Returning unmodified dataset.")
        return dataset, dataset_title, refresh_values


def _connect_to_vault(appui):
    """Check that the user has a token set.
    @params appui: The StarDrop application interface.
    @returns: True if a token has been set, False otherwise
    """
    global TOKEN
    logger.info('Connecting to vault')
    if TOKEN == '':
        logger.debug("No token set, changing token.")
        change_token(appui)
        if TOKEN == '':
            logger.debug("Unable to connect to vault due to missing token.")
            return False
    return True


def _get_batch_IDs_for_molecule(molecule):
    """Return the batch ids associated with the specified molecule.
    @params molecule: The molecule to extract the data from.
    @returns: The batch ids associated with the molecule.
    """
    logger.debug("Get batch IDs function started.")
    batch_ids = []
    for batch in molecule["batches"]:
        batch_ids.append((batch["id"], batch["molecule_batch_identifier"]))

    logger.debug(f"IDs retrieved: {batch_ids}")
    return batch_ids


def _get_project_IDs(vaultId, progress, progressvalue):
    """Find out the project(s) to search from this vault.
    @param: vaultId: The vault in which to locate projects.
    @param: progressvalue: The starting value when a progress bar is shown.
    @param: progressdialogue: A dialogue we'll be using to show progress.
    @returns: The selected project ids.
    """
    logger.debug("Get project IDs function started.")
    urlProject = cddconfig.url + f'/{vaultId[0]}/projects'
    progress.setWindowTitle(f"Finding projects in: {vaultId[1]}")
    progress.show()
    progress.setProgress(progressvalue)

    projects = {}
    try:
        results = _get_json_from_server(urlProject, TOKEN)
        for project in results:
            projects[project["name"]] = project["id"]
    except Exception as e:
        logger.exception(f"Unable to get project ID data from API: {str(e)}")

    logger.debug("Retrieved project ID data from API.")

    # Return found projects
    projectIDs = []
    if len(projects) == 0:
        progress.hide()
        logger.debug("Unable to find any project IDs")
        return []
    elif len(projects) == 1:
        logger.info('Searching project ' + list(projects.keys())[0])
        projectIDs.append(list(projects.values())[0])
    else:
        progress.hide()
        dlg = PythonWidgets.SimpleList()
        dlg.setAvailableItems(list(projects.keys()))
        dlg.setMultipleSelections()
        dlg.setWindowTitle("CDD Vault Access")
        dlg.setLabel("Select project(s) to search:")
        if not dlg.run():
            return USER_CANCELLED

        for item in dlg.getSelectedItems():
            projectIDs.append(projects[item])
        progress.show()
    logger.debug(f"Selected project IDs: {projectIDs}")
    return projectIDs


def _get_search_ID(vaultId, progress, progressvalue):
    """Find out the search to use form those available in this vault.
    @param: vaultId: The vault in which to locate projects.
    @param: progressvalue: The starting value when a progress bar is shown.
    @param: progressdialogue: A dialogue we'll be using to show progress.
    @returns: The selected search ids.
    """
    logger.debug("Get search ID function started.")
    urlSearch = cddconfig.url + f'/{vaultId[0]}/searches'
    progress.setWindowTitle(f"Finding saved searches in: {vaultId[1]}")
    progress.show()
    progress.setProgress(progressvalue)
    searches = {}

    try:
        results = _get_json_from_server(urlSearch, TOKEN)
        for search_result in results:
            searches[search_result["name"]] = search_result["id"]
    except Exception as e:
        logger.exception(f"Unable to get search ID data from API: {str(e)}")

    logger.debug("Retrieved search ID data from API.")

    if len(searches) == 0:
        progress.hide()
        logger.debug("Unable to find any search IDs.")
        return []
    # Ask the user to select/confirm a search to run
    dlg = PythonWidgets.SimpleList()
    dlg.setAvailableItems(list(searches.keys()))
    dlg.setWindowTitle("CDD Vault Access")
    dlg.setLabel("Select a saved search to run:")
    if not dlg.run():
        return USER_CANCELLED

    logger.debug("Search ID has been selected.")
    return (searches.get(dlg.getSelectedItem(), 0), dlg.getSelectedItem())


def _get_vault_ID(progress, progressvalue):
    """Find out the vault to access.
    @param: progressvalue: The starting value when a progress bar is shown.
    @param: progress: A dialog we'll be using to show progress.
    @returns: The selected vault id.
    """
    logger.debug("Get vault ID function started.")
    progress.setWindowTitle("Finding available vaults")
    progress.show()
    progress.setProgress(progressvalue)
    vaults = {}

    try:
        results = _get_json_from_server(cddconfig.url, TOKEN)
        # Find our Vault ID
        for vault in results:
            vaults[vault["name"]] = vault["id"]
    except Exception as e:
        logger.exception(f"Unable to get vault ID data from API: {str(e)}")

    # Return vault id
    if len(vaults) == 0:
        progress.hide()
        logger.debug("Unable to find any vault IDs.")
        return []
    elif len(vaults) == 1:
        logger.info('Using vault ' + list(vaults.keys())[0])
        logger.debug("Only one vault ID found, using it.")
        return (list(vaults.values())[0], list(vaults.keys())[0])
    else:
        progress.hide()
        dlg = PythonWidgets.SimpleList()
        dlg.setAvailableItems(list(vaults.keys()))
        dlg.setWindowTitle("CDD Vault Access")
        dlg.setLabel("Select a vault to search:")
        logger.debug("Multiple vault IDs found, giving the user the selection.")
        if not dlg.run():
            return USER_CANCELLED
        progress.show()
        logger.debug("User has selected the desired vault ID.")
        return (vaults[dlg.getSelectedItem()], dlg.getSelectedItem())


def _execute_search(project_ids, vault_id, search_id):
    """Perform the search API call.
    @params project_ids: The ids of the projects to search.
    @params vault_id: The id of the vault to search.
    @params search_id: The id of the search to perform.
    @returns: The result of the search API call.
    """
    logger.debug("Executing search.")
    params = {}
    if project_ids:
        pids = ','.join(map(str, project_ids))
        params['projects'] = pids
    if cddconfig.SDFILE_REQUIRED:
        params['format'] = 'sdf'
    if not params:
        params = None
    logger.debug(f"Search parameters constructed: {params}")

    url_export = cddconfig.url + f'/{vault_id}/searches/{search_id}'
    result = _get_json_from_server(url_export, TOKEN, params=params)
    logger.debug("API search call completed.")
    return result


def _check_search_status(appui, result, vault_id):
    """Check the status of the search until it is finished.
    @params appui: The StarDrop application interface.
    @params result: The result of the initial search API call.
    @params vault_id: The id of the vault.
    @returns: True if the search is complete, False if an issue occurs.
              And the id to retrieve the exported data from the search.
    """
    logger.debug("Checking search status.")
    export_id = result.get("id", 0)
    if export_id == 0:
        if appui:
            appui.showError("Unable to run the search")
        return False

    ready = False
    while not ready:
        url_check = cddconfig.url + f'/{vault_id}/export_progress/{export_id}'
        result = _get_json_from_server(url_check, TOKEN)
        status = result.get("status", "new")
        if status == "finished":
            ready = True
        elif status in ("new", "started"):
            time.sleep(1)
        else:
            logger.debug("Error retrieving search status from API.")
            break
    return ready, export_id


def _construct_dataset_from_search(appui, vault_id, export_id):
    """Construct a StarDrop dataset from the data retrieved from a search API call.
    @params appui: The StarDrop application interface.
    @params vault_id: The id of the vault.
    @params export_id: The id the retrieve the exported data from the search.
    @returns: A constructed StarDrop dataset, or None if there is an issue.
    """
    logger.debug("Constructing dataset from search data.")
    url_download = cddconfig.url + f'/{vault_id}/exports/{export_id}'
    result = _get_text_from_server(url_download, TOKEN)
    logger.debug("Retrieved data from API call.")
    dataset = None
    if result != '':
        if cddconfig.SDFILE_REQUIRED:
            logger.debug("SDFile required for dataset creation.")
            dataset = createDatasetFromSDFile(appui,
                                              result.splitlines(),
                                              cddconfig)
            for pos in list(reversed(range(dataset.columnSz()))):
                if dataset.columnName(pos) in ("SMILES", "ID"):
                    dataset.delColumn(pos)
        else:
            logger.debug("SDFile not required for dataset creation.")
            dataset = CSVHelper.createDataSet(result, cddconfig)

        if dataset.rowSz() == 0:
            logger.debug("Unable to successfully construct dataset from search data")
            return None
    return dataset


def create_image_data_entry(data, url):
    """Create an image data entry for a StarDrop dataset.
    @params data: The data to convert into a data entry.
    @params: url: The link to see the curve in CDD Vault
    @returns: The new entry.
    """
    try:
        if data:
            entry = process_plot_image(data.raw)
            url = app.Dataset.string(url)
            return (entry, url)
    except Exception as e:
        logger.exception(f"Unable to create image data entry: {str(e)}")
    return None


def _create_columns(dataset, protocol_info, entry_data, add_plot_data=False):
    """Create the new columns, and add them to the dataset.
    @params dataset: The StarDrop dataset to add the columns too.
    @params protocol_info: A dictionary of protocol names to protocol ids.
    @params entry_data: A list of entries to add to the dataset.
    @params add_plot_data: True if we want to include the plot images, False if not.
    @returns: The datatset that has been modified.
    """
    logger.debug(f"Creating columns for: {protocol_info}")
    for protocol in protocol_info:
        url_header = app.Dataset.meta(f"{protocol} URL")
        if add_plot_data:
            plot_header = app.Dataset.meta(protocol, app.image)

        row_val = dataset.rowSz()
        url_entries = []
        url_entries.extend([app.Dataset.createInvalidEntry(app.string) for i in range(row_val)])
        plot_entries = []
        if add_plot_data:
            plot_entries.extend([app.Dataset.createInvalidEntry(app.image) for i in range(row_val)])

        for data in entry_data:
            if add_plot_data:
                protocol_name = data[0]
                data_entry = data[1]
                entry_index = data[2]
                url_entry = data[3]
            else:
                protocol_name = data[0]
                entry_index = data[1]
                url_entry = data[2]

            if protocol_name == protocol_info[protocol]:
                if row_val > entry_index and url_entry:
                    url_entries[entry_index] = url_entry
                    if add_plot_data:
                        if data_entry:
                            plot_entries[entry_index] = data_entry

        dataset.insertColumn(url_header, url_entries, 1)
        if add_plot_data:
            dataset.insertColumn(plot_header, plot_entries, 1)

    logger.debug("Columns added to dataset.")
    return dataset


def _get_molecules(vault_id, params=None):
    """Get the molecule and it's batches back from the CDD vault
    @params vault_id: The unique ID of the vault to be searched.
    @returns: The molecule data from the CDD vault
    """
    logger.debug(f"Get molecules from vault: {vault_id}")
    urlMolecule = cddconfig.url + f'/{vault_id}/molecules'
    results = []
    try:
        results = _get_json_from_server(urlMolecule, TOKEN, params)
        ready, search_id = _check_search_status(None, results, vault_id)
        if ready:
            urlMoleculeRetrieve = cddconfig.url + f'/{vault_id}/exports/{search_id}'
            results = _get_json_from_server(urlMoleculeRetrieve, TOKEN)
            logger.debug("Molecule API call successful.")
        else:
            logger.error("Molecule search failed")
    except Exception as e:
        logger.exception(f"Unable to retrieve molecules from API: {str(e)}")
    return results


def _list_protocols(progress, progress_value, vault_id, molecule_ids, protocol_ids=None):
    """Get the desired protocol from a list of available protocols.
    @params progress: A dialog to show the progress of the process.
    @params progress_vaule: The starting value when a progress bar is shown.
    @params vault_id: The vault in which to locate protocols.
    @params molecule_ids: A list of molecule ids
    @returns: The protocol chosen by the user.
    """
    logger.debug("List protocols function started.")
    progress.setWindowTitle("Retrieving Protocol ID")
    progress.setProgress(progress_value)

    # Filter the results with the following parameters
    params = {}
    params['molecules'] = ','.join([str(m) for m in molecule_ids])

    protocol_info = _get_protocols(vault_id, params)
    protocols = {}
    for p in protocol_info["objects"]:
        protocols[p["name"]] = p["id"]
    logger.debug(f"Available protocols: {protocols}")

    if protocol_ids:
        logger.debug("Desired protocol IDs provided from refresh parameters, returning data.")
        return protocol_ids, protocols
    else:
        # If only one protocol return it, or ask the user to select which to use
        if len(protocols) == 0:
            logger.error("No protocols retrieved from API call.")
            return [], []
        elif len(protocols) == 1:
            logger.debug("Only one protocol returned, returning it.")
            return (list(protocols.values()), protocols)
        else:
            logger.debug("Multipe protocols returned, having the user selected which they want.")
            dlg = PythonWidgets.SimpleList()
            dlg.setWindowTitle("Select Protocol")
            dlg.setLabel("Select the protocol(s) to use:")
            dlg.setMultipleSelections()
            dlg.setAvailableItems(list(protocols.keys()))

        # Return a list of the selected protocols
        if not dlg.run():
            return USER_CANCELLED
        protocol_ids = []
        for item in dlg.getSelectedItems():
            protocol_ids.append(protocols[item])
        logger.debug(f"User selected the following protocols: {protocol_ids}")
        return protocol_ids, protocols


def _get_protocols(vault_id, params=None):
    """Get the protocols available within the specified vault.
    @params vault_id: The vault in which to locate protocols.
    @returns: The result from the server.
    """
    logger.debug("Get Protocols function started.")
    urlProtocol = cddconfig.url + f'/{vault_id}/protocols'
    results = []
    try:
        results = _get_json_from_server(urlProtocol, TOKEN, params)
    except Exception as e:
        logger.exception(f"Unable to retrieve protocol data: {str(e)}")
    return results


def _generate_images(progress, image_parameters):
    """Generate images for each of the response curves.
    @params progress: A progress dialog.
    @params image_parameters: A list of parameters used to retrieve the response curves
    @returns: A list of image entries to add to the dataset.
    """
    logger.debug("Generate images function started.")
    # Create progress bar
    t1 = time.time()
    progress.setWindowTitle("Retrieving plots")
    progress.setMaximum(len(image_parameters))
    progress.setProgress(0)

    count = 1
    image_results = {}
    thread_list = []
    for vault, batch, protocol, _, molecule, index, url, size in image_parameters:
        collector = ImageCollector(vault,
                                   molecule,
                                   batch,
                                   protocol,
                                   url,
                                   image_results,
                                   index,
                                   size)
        collector.start()
        thread_list.append(collector)
        # Execute 3 threads at a time
        if len(thread_list) == 3:
            for collector_thread in thread_list:
                collector_thread.join()
            thread_list = []
        progress.setProgress(count)
        count += 1
    # Execute remaining threads
    if thread_list:
        for collector_thread in thread_list:
            collector_thread.join()
    logger.debug(f"Generated images for {count} data entries.")

    entry_data = []
    for molecule in image_results:
        for protocol in image_results[molecule]:
            image_entry, link_entry, index = image_results[molecule][protocol]
            entry_data.append((protocol, image_entry, index, link_entry))
    t2 = time.time()
    logger.info(f"Plot retrieval time: {t2-t1}")
    return entry_data


def perform_plot_call(vault_id, batch_id, protocol_id, params):
    """Get dose-response curves/plots for a single batch, from the CDD vault.
    @params vault_id: The vault in which to locate a plot.
    @params batch_id: The batch in which to locate a plot.
    @params protocol_id: The porocol in which to locate a plot.
    @returns: The result back from the server
    """
    logger.debug("Performing plot call.")
    results = []
    url_plot = cddconfig.url + f'/{vault_id}/batches/{batch_id}/protocols/{protocol_id}/plot/query'
    try:
        results = stream_data_from_server(url_plot, TOKEN, params=params)
    except Exception as e:
        logger.exception(f"Unable to retrieve plot data from API: {str(e)}")
        return None
    logger.debug(f"Plot call successful for vault({vault_id}) - batch({batch_id}) - protocol({protocol_id})")
    return results


def process_plot_image(data):
    """Process the passed in data, into an StarDrop image data entry.
    @params data: The data to process into an image.
    @returns: The StarDrop data entry, or an invalid data entry.
    """
    logger.debug("Starting to process plot image.")
    filename = tempfile.mktemp()
    with open(filename, 'wb') as outfile:
        shutil.copyfileobj(data, outfile)

    try:
        entry = create_stardrop_entry_from_file(filename)
        logger.debug("Successfuly processed image data.")
        return entry
    except Exception as e:
        logger.exception(f"Unable to process image: {str(e)}")
        return None


def create_stardrop_entry_from_file(filename):
    """Open a PNG file and create a StarDrop data entry from the data.
    @params filename: The file to convert into a StarDrop data entry.
    @returns: A StarDrop data entry created using the file.
    """
    try:
        with open(filename, "rb") as fin:
            data = fin.read()
    except IOError:
        data = None
    if not data:
        logger.warning("Entry unsuccessfully created from file data")
        return app.Dataset.createInvalidEntry(app.image)
    logger.debug("Entry created successfully.")
    return create_stardrop_entry(data)


def create_stardrop_entry(data):
    """Create a StarDrop data entry from a byte array
    loaded from a file or downloaded.
    @params data: The data to convert into a StarDrop entry
    @returns: The converted StarDrop entry.
    """
    entry_data = []
    # version number for image provider
    entry_data.append(struct.pack("!i", 1))
    # version number for specific image provider
    entry_data.append(struct.pack("!i", 1))
    # length of raw data
    entry_data.append(struct.pack("!i", len(data)))
    # raw data itself
    entry_data.append(struct.pack("%ds" % len(data), data))
    # merge entries to a single byte array
    image_data = b''.join(entry_data)

    # create an entry of image type
    # (this assumes it is a URL so we need to hack the result around a bit
    entry = app.Dataset.image('')
    # get the entry as a list and extract the bit we need to manipulate
    entry_parts = list(entry)
    data_parts = list(entry_parts[2])
    # set the correct values on the data
    data_parts[2] = "RollingEmbeddedImageProvider"
    data_parts[3] = image_data

    # reassemble a data entry
    entry_parts[2] = tuple(data_parts)
    entry = tuple(entry_parts)
    return entry


def _get_json_from_server(url, token, params=None):
    """Make a request to the CDD Vault to get a JSON document.
    @param: url: The url of the resource being accessed.
    @param: token: The API token to be used, as a string.
    @returns: The data returned from the request, as a dictionary representing a JSON document.
    """
    txt = _get_text_from_server(url, token, params)
    return json.loads(txt)


def stream_data_from_server(url, token, params=None):
    """Make a request to the CDD Vault to stream raw data.
    @param: url: The url of the resource being accessed.
    @param: token: The API token to be used, as a string.
    @returns: The raw data returned from the request.
    """
    try:
        resp = requests.post(url, headers=_get_headers(token), json=params, stream=True)
        resp.raise_for_status()

    except HTTPError as e:
        logger.exception(str(e))
        return ''
    except Exception as e:
        logger.exception(e)
        return ''
    return resp


def _get_text_from_server(url, token, params=None):
    """Make a request to the CDD Vault to get raw data.
    @param: url: The url of the resource being accessed.
    @param: token: The API token to be used, as a string.
    @returns: The raw data returned from the request, as a string.
    """
    try:
        if params:
            url += "/query"
            resp = requests.post(url, headers=_get_headers(token), json=params)
        else:
            resp = requests.get(url, headers=_get_headers(token))
        resp.raise_for_status()
    except HTTPError as e:
        logger.exception(str(e))
        return ''
    except Exception as e:
        logger.exception(e)
        return ''
    return resp.text
